#include <WiFi.h>
#include <WebServer.h>

// Настройки точки доступа
const char* ssid = "MARS-BASE-1";        // Имя WiFi сети
const char* password = "12345678";       // Пароль (минимум 8 символов)

// Веб-сервер на порту 80
WebServer server(80);

// Пин светодиода
const int LED_PIN = 5;
bool ledState = false;

// HTML страница с интерфейсом управления
const char* htmlPage = R"=====(
<!DOCTYPE HTML>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Управление светодиодом МАРС-1</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            text-align: center;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            margin: 0;
        }
        .container {
            max-width: 400px;
            margin: 0 auto;
            background: rgba(255,255,255,0.1);
            padding: 30px;
            border-radius: 20px;
            backdrop-filter: blur(10px);
        }
        h1 {
            margin-bottom: 30px;
            font-size: 28px;
        }
        .status {
            font-size: 24px;
            margin: 20px 0;
            padding: 15px;
            background: rgba(0,0,0,0.2);
            border-radius: 10px;
        }
        .led-on {
            color: #4CAF50;
            text-shadow: 0 0 20px #4CAF50;
        }
        .led-off {
            color: #f44336;
            text-shadow: 0 0 20px #f44336;
        }
        button {
            font-size: 20px;
            padding: 15px 40px;
            margin: 10px;
            border: none;
            border-radius: 50px;
            cursor: pointer;
            transition: all 0.3s;
            color: white;
            font-weight: bold;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
        .btn-on {
            background: linear-gradient(45deg, #4CAF50, #8BC34A);
        }
        .btn-on:hover {
            transform: scale(1.05);
            box-shadow: 0 6px 20px rgba(76,175,80,0.4);
        }
        .btn-off {
            background: linear-gradient(45deg, #f44336, #E91E63);
        }
        .btn-off:hover {
            transform: scale(1.05);
            box-shadow: 0 6px 20px rgba(244,67,54,0.4);
        }
        .info {
            margin-top: 30px;
            padding: 15px;
            background: rgba(0,0,0,0.2);
            border-radius: 10px;
            font-size: 14px;
        }
        .emoji {
            font-size: 50px;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 База МАРС-1</h1>
        <h2>Система освещения</h2>
        
        <div class="emoji" id="emoji">💡</div>
        
        <div class="status">
            Светодиод: <span id="ledStatus" class="led-off">ВЫКЛЮЧЕН</span>
        </div>
        
        <div>
            <button class="btn-on" onclick="controlLED('on')">✨ ВКЛЮЧИТЬ</button>
            <button class="btn-off" onclick="controlLED('off')">🌑 ВЫКЛЮЧИТЬ</button>
        </div>
        
        <div class="info">
            📡 Система управления освещением<br>
            🌐 IP адрес: 192.168.4.1<br>
            🔐 Защищённое соединение
        </div>
    </div>
    
    <script>
        function controlLED(action) {
            fetch('/LED_' + action)
                .then(response => response.text())
                .then(data => {
                    updateStatus(action === 'on');
                });
        }
        
        function updateStatus(isOn) {
            const status = document.getElementById('ledStatus');
            const emoji = document.getElementById('emoji');
            if(isOn) {
                status.textContent = 'ВКЛЮЧЕН';
                status.className = 'led-on';
                emoji.textContent = '💡';
            } else {
                status.textContent = 'ВЫКЛЮЧЕН';
                status.className = 'led-off';
                emoji.textContent = '🌑';
            }
        }
        
        // Проверяем статус при загрузке
        fetch('/status')
            .then(response => response.text())
            .then(data => {
                updateStatus(data === 'on');
            });
    </script>
</body>
</html>
)=====";

void setup() {
  Serial.begin(115200);
  Serial.println("\n===== СИСТЕМА УПРАВЛЕНИЯ МАРС-1 =====");
  
  // Настройка пина светодиода
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LOW);
  
  // Создание точки доступа
  Serial.println("Создание точки доступа...");
  WiFi.mode(WIFI_AP);
  WiFi.softAP(ssid, password);
  
  // Получение IP адреса точки доступа
  IPAddress IP = WiFi.softAPIP();
  Serial.print("Точка доступа создана!\n");
  Serial.print("Имя сети: ");
  Serial.println(ssid);
  Serial.print("Пароль: ");
  Serial.println(password);
  Serial.print("IP адрес: ");
  Serial.println(IP);
  
  // Настройка маршрутов веб-сервера
  server.on("/", handleRoot);
  server.on("/LED_on", handleLedOn);
  server.on("/LED_off", handleLedOff);
  server.on("/status", handleStatus);
  
  // Запуск веб-сервера
  server.begin();
  Serial.println("Веб-сервер запущен!");
  Serial.println("\nПодключитесь к WiFi: " + String(ssid));
  Serial.println("Откройте браузер: http://192.168.4.1");
}

void loop() {
  server.handleClient();
}

// Обработчик главной страницы
void handleRoot() {
  server.send(200, "text/html", htmlPage);
}

// Обработчик включения светодиода
void handleLedOn() {
  ledState = true;
  digitalWrite(LED_PIN, HIGH);
  server.send(200, "text/plain", "LED включен");
  Serial.println("Светодиод ВКЛЮЧЕН");
}

// Обработчик выключения светодиода
void handleLedOff() {
  ledState = false;
  digitalWrite(LED_PIN, LOW);
  server.send(200, "text/plain", "LED выключен");
  Serial.println("Светодиод ВЫКЛЮЧЕН");
}

// Обработчик статуса светодиода
void handleStatus() {
  server.send(200, "text/plain", ledState ? "on" : "off");
}
