#include <WiFi.h>
#include <WebServer.h>

// Настройки точки доступа
const char* ssid = "MARS-PIANO";        // Имя WiFi сети
const char* password = "12345678";      // Пароль (минимум 8 символов)

WebServer server(80);

// HTML страница с пианино
const char* htmlPage = R"=====(
<!DOCTYPE HTML>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Марсианское пианино</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 30px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 800px;
            width: 100%;
        }
        
        h1 {
            text-align: center;
            color: #333;
            margin-bottom: 10px;
            font-size: 32px;
            font-weight: 600;
        }
        
        .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
            font-size: 16px;
        }
        
        .piano {
            display: flex;
            justify-content: center;
            align-items: flex-end;
            margin: 30px 0;
            position: relative;
            height: 200px;
        }
        
        .key {
            position: relative;
            cursor: pointer;
            transition: all 0.1s ease;
            user-select: none;
        }
        
        .white-key {
            width: 60px;
            height: 180px;
            background: linear-gradient(180deg, #fff 0%, #f5f5f5 100%);
            border: 1px solid #ddd;
            border-radius: 0 0 8px 8px;
            margin: 0 2px;
            box-shadow: 0 5px 10px rgba(0,0,0,0.1);
            display: flex;
            align-items: flex-end;
            justify-content: center;
            padding-bottom: 15px;
            font-weight: 600;
            font-size: 18px;
            color: #666;
        }
        
        .white-key:hover {
            background: linear-gradient(180deg, #f8f8f8 0%, #e8e8e8 100%);
        }
        
        .white-key:active,
        .white-key.active {
            transform: translateY(2px);
            background: linear-gradient(180deg, #e0e0e0 0%, #d0d0d0 100%);
            box-shadow: 0 2px 5px rgba(0,0,0,0.1) inset;
        }
        
        .black-key {
            position: absolute;
            width: 40px;
            height: 120px;
            background: linear-gradient(180deg, #333 0%, #000 100%);
            border-radius: 0 0 6px 6px;
            box-shadow: 0 5px 10px rgba(0,0,0,0.3);
            z-index: 2;
            top: 0;
            color: #fff;
            display: flex;
            align-items: flex-end;
            justify-content: center;
            padding-bottom: 10px;
            font-size: 14px;
            font-weight: 600;
        }
        
        .black-key:hover {
            background: linear-gradient(180deg, #444 0%, #111 100%);
        }
        
        .black-key:active,
        .black-key.active {
            transform: translateY(2px);
            background: linear-gradient(180deg, #222 0%, #000 100%);
            box-shadow: 0 2px 5px rgba(0,0,0,0.3) inset;
        }
        
        #key-CS { left: 85px; }
        #key-DS { left: 149px; }
        #key-FS { left: 277px; }
        #key-GS { left: 341px; }
        #key-AS { left: 405px; }
        
        .info {
            background: rgba(103, 126, 234, 0.1);
            border-radius: 15px;
            padding: 20px;
            margin-top: 30px;
            text-align: center;
            color: #666;
        }
        
        .status {
            display: inline-block;
            padding: 8px 16px;
            background: #4CAF50;
            color: white;
            border-radius: 20px;
            font-size: 14px;
            margin-top: 10px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }
        
        .emoji {
            font-size: 40px;
            margin-bottom: 10px;
            animation: float 3s ease-in-out infinite;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }
        
        @media (max-width: 600px) {
            .white-key {
                width: 40px;
                height: 140px;
                font-size: 14px;
            }
            .black-key {
                width: 30px;
                height: 90px;
                font-size: 12px;
            }
            #key-CS { left: 55px; }
            #key-DS { left: 99px; }
            #key-FS { left: 187px; }
            #key-GS { left: 231px; }
            #key-AS { left: 275px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="emoji">🎹</div>
        <h1>Марсианское пианино</h1>
        <p class="subtitle">Нажмите на клавиши, чтобы играть музыку</p>
        
        <div class="piano">
            <div class="key white-key" onclick="playNote('C')" id="key-C">C</div>
            <div class="key white-key" onclick="playNote('D')" id="key-D">D</div>
            <div class="key white-key" onclick="playNote('E')" id="key-E">E</div>
            <div class="key white-key" onclick="playNote('F')" id="key-F">F</div>
            <div class="key white-key" onclick="playNote('G')" id="key-G">G</div>
            <div class="key white-key" onclick="playNote('A')" id="key-A">A</div>
            <div class="key white-key" onclick="playNote('B')" id="key-B">B</div>
            
            <div class="key black-key" onclick="playNote('CS')" id="key-CS">C#</div>
            <div class="key black-key" onclick="playNote('DS')" id="key-DS">D#</div>
            <div class="key black-key" onclick="playNote('FS')" id="key-FS">F#</div>
            <div class="key black-key" onclick="playNote('GS')" id="key-GS">G#</div>
            <div class="key black-key" onclick="playNote('AS')" id="key-AS">A#</div>
        </div>
        
        <div class="info">
            <p>🎵 Музыкальная система базы МАРС-1</p>
            <div class="status">📡 Подключено к ESP32</div>
        </div>
    </div>
    
    <script>
        function playNote(note) {
            // Визуальная обратная связь
            const key = document.getElementById('key-' + note);
            key.classList.add('active');
            setTimeout(() => key.classList.remove('active'), 200);
            
            // Отправка запроса на ESP32
            fetch('/play?note=' + note)
                .then(response => response.text())
                .then(data => console.log('Played:', note))
                .catch(error => console.error('Error:', error));
        }
        
        // Поддержка клавиатуры
        document.addEventListener('keydown', function(event) {
            const keyMap = {
                'a': 'C',
                's': 'D',
                'd': 'E',
                'f': 'F',
                'g': 'G',
                'h': 'A',
                'j': 'B',
                'w': 'CS',
                'e': 'DS',
                't': 'FS',
                'y': 'GS',
                'u': 'AS'
            };
            
            const note = keyMap[event.key.toLowerCase()];
            if (note && !event.repeat) {
                playNote(note);
            }
        });
    </script>
</body>
</html>
)=====";

void setup() {
  Serial.begin(115200);
  Serial.println("\n===== МАРСИАНСКОЕ ПИАНИНО =====");
  
  // Настройка пина зуммера
  pinMode(23, OUTPUT);
  digitalWrite(23, LOW);
  
  // Создание точки доступа
  Serial.println("Создание точки доступа...");
  WiFi.mode(WIFI_AP);
  WiFi.softAP(ssid, password);
  
  // Получение IP адреса
  IPAddress IP = WiFi.softAPIP();
  Serial.print("Точка доступа создана!\n");
  Serial.print("Имя сети: ");
  Serial.println(ssid);
  Serial.print("Пароль: ");
  Serial.println(password);
  Serial.print("IP адрес: ");
  Serial.println(IP);
  
  // Настройка веб-сервера
  server.on("/", handleRoot);
  server.on("/play", handlePlayNote);
  
  // Запуск сервера
  server.begin();
  Serial.println("Сервер запущен!");
  Serial.println("\nПодключитесь к WiFi: " + String(ssid));
  Serial.println("Откройте браузер: http://192.168.4.1");
  Serial.println("Играйте на пианино!");
}

void loop() {
  server.handleClient();
}

void handleRoot() {
  server.send(200, "text/html", htmlPage);
}

void handlePlayNote() {
  if (server.hasArg("note")) {
    String note = server.arg("note");
    Serial.print("Играет нота: ");
    Serial.println(note);
    
    // Основные ноты
    if (note == "C") {
      tone(23, 261.63, 500);  // До
    } else if (note == "D") {
      tone(23, 293.66, 500);  // Ре
    } else if (note == "E") {
      tone(23, 329.63, 500);  // Ми
    } else if (note == "F") {
      tone(23, 349.23, 500);  // Фа
    } else if (note == "G") {
      tone(23, 392.00, 500);  // Соль
    } else if (note == "A") {
      tone(23, 440.00, 500);  // Ля
    } else if (note == "B") {
      tone(23, 493.88, 500);  // Си
    }
    // Черные клавиши (диезы)
    else if (note == "CS") {
      tone(23, 277.18, 500);  // До-диез
    } else if (note == "DS") {
      tone(23, 311.13, 500);  // Ре-диез
    } else if (note == "FS") {
      tone(23, 369.99, 500);  // Фа-диез
    } else if (note == "GS") {
      tone(23, 415.30, 500);  // Соль-диез
    } else if (note == "AS") {
      tone(23, 466.16, 500);  // Ля-диез
    }
  }
  server.send(200, "text/plain", "OK");
}
