#include <WiFi.h>
#include <WebServer.h>
#include <ESP32Servo.h>

// Настройки точки доступа
const char* ssid = "MARS-DOOR";         // Имя WiFi сети
const char* password = "12345678";      // Пароль (минимум 8 символов)

// Веб-сервер на порту 80
WebServer server(80);

// Настройки серводвигателя
Servo doorServo;
const int SERVO_PIN = 18;
const int DOOR_CLOSED = 5;
const int DOOR_OPEN = 100;
bool doorState = false; // false = закрыта, true = открыта

// HTML страница с интерфейсом управления
const char* htmlPage = R"=====(
<!DOCTYPE HTML>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Система управления шлюзом МАРС-1</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 30px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 500px;
            width: 100%;
        }
        
        h1 {
            text-align: center;
            color: #333;
            margin-bottom: 10px;
            font-size: 32px;
            font-weight: 600;
        }
        
        .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
            font-size: 16px;
        }
        
        .door-container {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            border-radius: 20px;
            padding: 30px;
            margin: 30px 0;
            text-align: center;
        }
        
        .door-icon {
            font-size: 80px;
            margin: 20px 0;
            animation: float 3s ease-in-out infinite;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }
        
        .status {
            font-size: 24px;
            margin: 20px 0;
            padding: 15px;
            background: rgba(255, 255, 255, 0.8);
            border-radius: 10px;
            font-weight: 600;
        }
        
        .status-open {
            color: #4CAF50;
            text-shadow: 0 0 20px rgba(76, 175, 80, 0.5);
        }
        
        .status-closed {
            color: #f44336;
            text-shadow: 0 0 20px rgba(244, 67, 54, 0.5);
        }
        
        .button-container {
            display: flex;
            gap: 20px;
            justify-content: center;
            margin: 30px 0;
        }
        
        button {
            font-size: 18px;
            padding: 15px 30px;
            border: none;
            border-radius: 50px;
            cursor: pointer;
            transition: all 0.3s ease;
            color: white;
            font-weight: bold;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
            position: relative;
            overflow: hidden;
        }
        
        button:before {
            content: "";
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.3);
            transform: translate(-50%, -50%);
            transition: width 0.6s, height 0.6s;
        }
        
        button:hover:before {
            width: 300px;
            height: 300px;
        }
        
        .btn-open {
            background: linear-gradient(45deg, #4CAF50, #8BC34A);
        }
        
        .btn-open:hover {
            transform: scale(1.05);
            box-shadow: 0 6px 20px rgba(76, 175, 80, 0.4);
        }
        
        .btn-close {
            background: linear-gradient(45deg, #f44336, #E91E63);
        }
        
        .btn-close:hover {
            transform: scale(1.05);
            box-shadow: 0 6px 20px rgba(244, 67, 54, 0.4);
        }
        
        button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: scale(1);
        }
        
        .info {
            background: rgba(103, 126, 234, 0.1);
            border-radius: 15px;
            padding: 20px;
            margin-top: 30px;
            text-align: center;
            color: #666;
        }
        
        .connection-status {
            display: inline-block;
            padding: 8px 16px;
            background: #4CAF50;
            color: white;
            border-radius: 20px;
            font-size: 14px;
            margin-top: 10px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }
        
        .security-warning {
            background: rgba(255, 193, 7, 0.1);
            border-left: 4px solid #FFC107;
            padding: 15px;
            border-radius: 10px;
            margin: 20px 0;
            color: #856404;
        }
        
        .loading {
            display: none;
            text-align: center;
            margin: 20px 0;
        }
        
        .loading.active {
            display: block;
        }
        
        .spinner {
            border: 3px solid rgba(0, 0, 0, 0.1);
            border-radius: 50%;
            border-top: 3px solid #667eea;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @media (max-width: 480px) {
            h1 { font-size: 24px; }
            .door-icon { font-size: 60px; }
            button { padding: 12px 25px; font-size: 16px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 База МАРС-1</h1>
        <p class="subtitle">Система управления воздушным шлюзом</p>
        
        <div class="door-container">
            <div class="door-icon" id="doorIcon">🚪</div>
            <div class="status" id="doorStatus">
                Шлюз: <span id="statusText" class="status-closed">ЗАКРЫТ</span>
            </div>
            
            <div class="loading" id="loading">
                <div class="spinner"></div>
                <p style="margin-top: 10px; color: #666;">Выполнение команды...</p>
            </div>
            
            <div class="button-container">
                <button class="btn-open" onclick="controlDoor('open')" id="btnOpen">
                    🔓 ОТКРЫТЬ ШЛЮЗ
                </button>
                <button class="btn-close" onclick="controlDoor('close')" id="btnClose">
                    🔒 ЗАКРЫТЬ ШЛЮЗ
                </button>
            </div>
        </div>
        
        <div class="security-warning">
            <strong>⚠️ Внимание!</strong><br>
            Перед открытием шлюза убедитесь в герметичности скафандра и наличии кислорода.
        </div>
        
        <div class="info">
            <p>🛡️ Система безопасности марсианской базы</p>
            <div class="connection-status">📡 Подключено к ESP32</div>
        </div>
    </div>
    
    <script>
        let isProcessing = false;
        let currentState = 'closed';
        
        function controlDoor(action) {
            if (isProcessing) return;
            
            isProcessing = true;
            document.getElementById('loading').classList.add('active');
            document.getElementById('btnOpen').disabled = true;
            document.getElementById('btnClose').disabled = true;
            
            fetch('/door_' + action)
                .then(response => response.json())
                .then(data => {
                    updateDoorStatus(data.state);
                    isProcessing = false;
                    document.getElementById('loading').classList.remove('active');
                    updateButtons();
                })
                .catch(error => {
                    console.error('Error:', error);
                    isProcessing = false;
                    document.getElementById('loading').classList.remove('active');
                    updateButtons();
                });
        }
        
        function updateDoorStatus(state) {
            currentState = state;
            const statusText = document.getElementById('statusText');
            const doorIcon = document.getElementById('doorIcon');
            
            if (state === 'open') {
                statusText.textContent = 'ОТКРЫТ';
                statusText.className = 'status-open';
                doorIcon.textContent = '🚪';
                doorIcon.style.transform = 'rotate(-25deg)';
            } else 
                statusText.textContent = 'ЗАКРЫТО';
                statusText.className = 'status-closed';
                doorIcon.textContent = '🚪';
                doorIcon.style.transform = 'rotate(0deg)';
            }
        }
        
        function updateButtons() {
            document.getElementById('btnOpen').disabled = currentState === 'open';
            document.getElementById('btnClose').disabled = currentState === 'closed';
        }
        
        // Проверка статуса при загрузке
        fetch('/status')
            .then(response => response.json())
            .then(data => {
                updateDoorStatus(data.state);
                updateButtons();
            });
        
        // Периодическое обновление статуса
        setInterval(() => {
            if (!isProcessing) {
                fetch('/status')
                    .then(response => response.json())
                    .then(data => {
                        updateDoorStatus(data.state);
                        updateButtons();
                    });
            }
        }, 5000);
    </script>
</body>
</html>
)=====";

void setup() {
  Serial.begin(115200);
  Serial.println("\n===== СИСТЕМА УПРАВЛЕНИЯ ШЛЮЗОМ МАРС-1 =====");
  
  // Настройка серводвигателя
  doorServo.attach(SERVO_PIN, 500, 2500);
  doorServo.write(DOOR_CLOSED);
  delay(500);
  Serial.println("Шлюз закрыт");
  
  // Создание точки доступа
  Serial.println("Создание точки доступа...");
  WiFi.mode(WIFI_AP);
  WiFi.softAP(ssid, password);
  
  // Получение IP адреса
  IPAddress IP = WiFi.softAPIP();
  Serial.print("Точка доступа создана!\n");
  Serial.print("Имя сети: ");
  Serial.println(ssid);
  Serial.print("Пароль: ");
  Serial.println(password);
  Serial.print("IP адрес: ");
  Serial.println(IP);
  
  // Настройка веб-сервера
  server.on("/", handleRoot);
  server.on("/door_open", handleDoorOpen);
  server.on("/door_close", handleDoorClose);
  server.on("/status", handleStatus);
  
  // Запуск сервера
  server.begin();
  Serial.println("Веб-сервер запущен!");
  Serial.println("\nПодключитесь к WiFi: " + String(ssid));
  Serial.println("Откройте браузер: http://192.168.4.1");
  Serial.println("Управляйте шлюзом!");
}

void loop() {
  server.handleClient();
}

// Обработчик главной страницы
void handleRoot() {
  server.send(200, "text/html", htmlPage);
}

// Обработчик открытия двери
void handleDoorOpen() {
  doorServo.write(DOOR_OPEN);
  doorState = true;
  Serial.println("Шлюз ОТКРЫТ");
  
  String json = "{\"state\":\"open\",\"message\":\"Шлюз открыт\"}";
  server.send(200, "application/json", json);
}

// Обработчик закрытия двери
void handleDoorClose() {
  doorServo.write(DOOR_CLOSED);
  doorState = false;
  Serial.println("Шлюз ЗАКРЫТ");
  
  String json = "{\"state\":\"closed\",\"message\":\"Шлюз закрыт\"}";
  server.send(200, "application/json", json);
}

// Обработчик статуса
void handleStatus() {
  String state = doorState ? "open" : "closed";
  String json = "{\"state\":\"" + state + "\"}";
  server.send(200, "application/json", json);
}
