// Система управления базой МАРС-1
#include <Arduino.h>
#include <WiFi.h>
#include <WebServer.h>
#include <ESP32Servo.h>
#include <Wire.h>
#include <Adafruit_PN532.h>
#include <DHT.h>

// LCD с русским языком
#define _LCD_TYPE 1
#include <LCD_1602_RUS_ALL.h>

// WiFi настройки - ESP32 как точка доступа
const char* ssid = "MARS-BASE-1";
const char* password = "12345678";

WebServer server(80);  // WebServer вместо WiFiServer

// NFC модуль PN532
#define SDA_PIN 21
#define SCL_PIN 22
Adafruit_PN532 nfc(SDA_PIN, SCL_PIN);
String nfc_str = "";
String last_card_user = "";

// LCD дисплей
LCD_1602_RUS lcd(0x27, 16, 2);

// Сервопривод шлюза
Servo servo_18;
const int door_close = 5;
const int door_open = 100;
bool door_status = false;

// DHT датчик
DHT dht25(25, DHT11);
float temperature = 0;
float humidity = 0;

// Пины и переменные
const int buzzerPin = 23;
const int ledPin = 5;
const int pirPin = 14;
const int gasPin = 32;
const int buttonPin = 26;

// База данных карт доступа
String commander_uid = "c3fd53fa";
String engineer_uid = "d778982c";

// Состояния системы
bool led_status = false;
bool pir_detected = false;
int gas_value = 0;
bool gas_alarm = false;
bool auto_mode = true;
bool wifi_mode = false;
unsigned long lastUpdate = 0;
unsigned long wifi_time = 0;
int wifi_mark = 0;

// Музыкальные ноты
#define C4 262
#define D4 294
#define E4 330
#define F4 350
#define G4 393
#define A4 441
#define B4 495
#define C5 525
#define N 0

// Мелодии
int tune0[] = { C4, N, C4, G4, N, G4, A4, N, A4, G4, N, F4, N, F4, E4, N, E4, D4, N, D4, C4 };
float durt0[] = { 0.99, 0.01, 1, 0.99, 0.01, 1, 0.99, 0.01, 1, 1.95, 0.05, 0.99, 0.01, 1, 0.99, 0.01, 1, 0.99, 0.01, 1, 2 };
int length0;

// HTML страница
const char index_html[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>МАРС-1 Центр Управления</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1e3c72, #2a5298);
            min-height: 100vh;
            color: #fff;
            padding: 20px;
        }
        .header {
            background: rgba(0,0,0,0.3);
            padding: 20px;
            text-align: center;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        h1 { color: #ff6b6b; text-shadow: 2px 2px 4px rgba(0,0,0,0.5); }
        .dashboard {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
        }
        .card {
            background: rgba(255,255,255,0.1);
            backdrop-filter: blur(10px);
            border-radius: 15px;
            padding: 20px;
            border: 1px solid rgba(255,255,255,0.2);
        }
        .card h2 {
            color: #ffd93d;
            margin-bottom: 15px;
        }
        .sensor-value {
            font-size: 1.5em;
            font-weight: bold;
            margin: 10px 0;
            text-align: center;
            padding: 10px;
            background: rgba(0,0,0,0.3);
            border-radius: 10px;
        }
        .btn {
            width: 100%;
            padding: 10px;
            margin: 5px 0;
            border: none;
            border-radius: 5px;
            font-weight: bold;
            cursor: pointer;
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
        }
        .btn:hover { opacity: 0.8; }
        .status-on { color: #00ff00; }
        .status-off { color: #ff0000; }
    </style>
</head>
<body>
    <div class="header">
        <h1>🚀 БАЗА МАРС-1</h1>
        <p>Центр управления системами</p>
    </div>
    
    <div class="dashboard">
        <div class="card">
            <h2>🌡️ Климат</h2>
            <div class="sensor-value">
                T: <span id="temp">--</span>°C | H: <span id="humid">--</span>%
            </div>
        </div>
        
        <div class="card">
            <h2>🚪 Шлюз</h2>
            <button class="btn" onclick="control('door_open')">Открыть</button>
            <button class="btn" onclick="control('door_close')">Закрыть</button>
        </div>
        
        <div class="card">
            <h2>💡 Освещение</h2>
            <button class="btn" onclick="control('LED_toggle')">Вкл/Выкл</button>
        </div>
        
        <div class="card">
            <h2>🔐 Безопасность</h2>
            <div class="sensor-value">
                Газ: <span id="gas">--</span><br>
                PIR: <span id="pir" class="status-off">НЕТ</span>
            </div>
        </div>
        
        <div class="card">
            <h2>🎵 Музыка</h2>
            <button class="btn" onclick="control('Music_01')">Мелодия 1</button>
            <button class="btn" onclick="control('music_stop')">Стоп</button>
        </div>
    </div>
    
    <script>
        function control(cmd) {
            fetch('/' + cmd)
                .then(response => response.text())
                .then(data => console.log(data));
        }
        
        function updateData() {
            fetch('/temperature_req').then(r => r.text()).then(data => {
                document.getElementById('temp').innerText = data || '--';
            });
            
            fetch('/humidity_req').then(r => r.text()).then(data => {
                document.getElementById('humid').innerText = data || '--';
            });
            
            fetch('/gas_req').then(r => r.text()).then(data => {
                document.getElementById('gas').innerText = data || '--';
            });
            
            fetch('/PIR_req').then(r => r.text()).then(data => {
                let pirEl = document.getElementById('pir');
                if(data == '1') {
                    pirEl.innerText = 'ДВИЖЕНИЕ';
                    pirEl.className = 'status-on';
                } else {
                    pirEl.innerText = 'НЕТ';
                    pirEl.className = 'status-off';
                }
            });
        }
        
        setInterval(updateData, 2000);
        updateData();
    </script>
</body>
</html>
)rawliteral";

// Обработчики HTTP запросов
void handleRoot() {
  server.send(200, "text/html", index_html);
  wifi_mode = true;
  wifi_mark = 1;
  wifi_time = millis();
  lcd.setCursor(0, 1);
  lcd.print("WiFi режим      ");
}

void handleLedOn() {
  digitalWrite(ledPin, HIGH);
  led_status = true;
  server.send(200, "text/plain", "LED ON");
}

void handleLedOff() {
  digitalWrite(ledPin, LOW);
  led_status = false;
  server.send(200, "text/plain", "LED OFF");
}

void handleLedToggle() {
  led_status = !led_status;
  digitalWrite(ledPin, led_status);
  server.send(200, "text/plain", led_status ? "LED ON" : "LED OFF");
}

void handleDoorOpen() {
  servo_18.write(door_open);
  door_status = true;
  server.send(200, "text/plain", "DOOR OPEN");
}

void handleDoorClose() {
  servo_18.write(door_close);
  door_status = false;
  server.send(200, "text/plain", "DOOR CLOSE");
}

void handleTemperature() {
  server.send(200, "text/plain", String(int(temperature)));
}

void handleHumidity() {
  server.send(200, "text/plain", String(int(humidity)));
}

void handleGas() {
  server.send(200, "text/plain", String(gas_value));
}

void handlePIR() {
  server.send(200, "text/plain", String(pir_detected));
}

void handleMusic() {
  // Отключаем сервопривод временно
  servo_18.detach();
  
  for (int x = 0; x < length0; x++) {
    tone(buzzerPin, tune0[x]);
    delay(500 * durt0[x]);
    noTone(buzzerPin);
  }
  
  // Включаем сервопривод обратно
  servo_18.attach(18, 500, 2500);
  server.send(200, "text/plain", "MUSIC OK");
}

void handleMusicStop() {
  noTone(buzzerPin);
  server.send(200, "text/plain", "MUSIC STOP");
}

// Функция кнопки
void checkButton() {
  if (digitalRead(buttonPin) == LOW) {
    lcd.setCursor(0, 1);
    lcd.print("Кнопка нажата   ");
    
    servo_18.detach();  // Отключаем servo перед музыкой
    for (int x = 0; x < length0; x++) {
      tone(buzzerPin, tune0[x]);
      delay(500 * durt0[x]);
      noTone(buzzerPin);
    }
    servo_18.attach(18, 500, 2500);  // Включаем обратно
  }
}

// Проверка PIR
void checkPIR() {
  pir_detected = digitalRead(pirPin);
  
  if (auto_mode) {
    if (pir_detected && !led_status) {
      digitalWrite(ledPin, HIGH);
      led_status = true;
      lcd.setCursor(0, 1);
      lcd.print("Движение-свет   ");
    } else if (!pir_detected && led_status) {
      digitalWrite(ledPin, LOW);
      led_status = false;
      lcd.setCursor(0, 1);
      lcd.print("Движения нет    ");
    }
  }
}

// Чтение температуры и влажности
void readClimate() {
  float t = dht25.readTemperature();
  float h = dht25.readHumidity();
  
  if (!isnan(t)) temperature = t;
  if (!isnan(h)) humidity = h;
  
  // Отображение на LCD
  char tempStr[6];
  char humStr[6];
  dtostrf(temperature, 4, 1, tempStr);
  dtostrf(humidity, 4, 1, humStr);
  
  lcd.setCursor(0, 0);
  lcd.print("T:");
  lcd.print(tempStr);
  lcd.print(" H:");
  lcd.print(humStr);
}

// Проверка газа
void checkGas() {
  gas_value = analogRead(gasPin);
  
  if (gas_value >= 2000 && !gas_alarm) {
    gas_alarm = true;
    lcd.setCursor(0, 1);
    lcd.print("ТРЕВОГА! ГАЗ!   ");
    
    tone(buzzerPin, 1000);
    delay(1000);
    noTone(buzzerPin);
  } else if (gas_value < 2000 && gas_alarm) {
    gas_alarm = false;
    lcd.setCursor(0, 1);
    lcd.print("Газ в норме     ");
  }
}

// Проверка NFC
void checkNFC() {
  uint8_t uid[7];
  uint8_t uidLength;
  
  if (!nfc.readPassiveTargetID(PN532_MIFARE_ISO14443A, uid, &uidLength, 100)) {
    return;
  }
  
  // Формируем UID
  nfc_str = "";
  for (uint8_t i = 0; i < uidLength; i++) {
    if (uid[i] < 16) nfc_str += "0";
    nfc_str += String(uid[i], HEX);
  }
  
  Serial.print("UID: ");
  Serial.println(nfc_str);
  
  // Проверка доступа
  if (nfc_str == commander_uid) {
    last_card_user = "КОМАНДИР";
    lcd.setCursor(0, 1);
    lcd.print("Командир-доступ ");
    
    servo_18.write(door_open);
    door_status = true;
    
    tone(buzzerPin, 2000);
    delay(100);
    noTone(buzzerPin);
    delay(50);
    tone(buzzerPin, 2000);
    delay(100);
    noTone(buzzerPin);
    
    delay(2000);
    
    servo_18.write(door_close);
    door_status = false;
    lcd.setCursor(0, 1);
    lcd.print("Дверь закрыта   ");
    
  } else if (nfc_str == engineer_uid) {
    last_card_user = "ИНЖЕНЕР";
    lcd.setCursor(0, 1);
    lcd.print("Инженер-доступ  ");
    
    servo_18.write(door_open);
    door_status = true;
    
    tone(buzzerPin, 2000);
    delay(100);
    noTone(buzzerPin);
    delay(50);
    tone(buzzerPin, 2000);
    delay(100);
    noTone(buzzerPin);
    
    delay(2000);
    
    servo_18.write(door_close);
    door_status = false;
    lcd.setCursor(0, 1);
    lcd.print("Дверь закрыта   ");
    
  } else {
    last_card_user = "НЕИЗВЕСТНЫЙ";
    lcd.setCursor(0, 1);
    lcd.print("Доступ запрещен!");
    
    tone(buzzerPin, 500);
    delay(1000);
    noTone(buzzerPin);
  }
}

void setup() {
  Serial.begin(115200);
  Wire.begin();
  
  // LCD инициализация
  lcd.init();
  lcd.backlight();
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("МАРС-1 БАЗА");
  lcd.setCursor(0, 1);
  lcd.print("Запуск...");
  delay(2000);
  
  // Пины
  pinMode(ledPin, OUTPUT);
  pinMode(pirPin, INPUT);
  pinMode(gasPin, INPUT);
  pinMode(buttonPin, INPUT_PULLUP);
  pinMode(buzzerPin, OUTPUT);
  
  // NFC инициализация
  nfc.begin();
  uint32_t versiondata = nfc.getFirmwareVersion();
  if (!versiondata) {
    Serial.println("NFC модуль не найден");
    lcd.setCursor(0, 1);
    lcd.print("NFC не найден   ");
  } else {
    nfc.SAMConfig();
    Serial.println("NFC готов");
  }
  
  // DHT инициализация
  dht25.begin();
  
  // Сервопривод
  servo_18.attach(18, 500, 2500);
  servo_18.write(door_close);
  delay(500);
  
  // Музыка
  length0 = sizeof(tune0) / sizeof(tune0[0]);
  
  // WiFi точка доступа
  WiFi.mode(WIFI_AP);
  WiFi.softAP(ssid, password);
  
  IPAddress IP = WiFi.softAPIP();
  Serial.print("AP IP: ");
  Serial.println(IP);
  
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("WiFi:MARS-BASE-1");
  lcd.setCursor(0, 1);
  lcd.print(IP.toString());
  delay(3000);
  
  // Настройка маршрутов веб-сервера
  server.on("/", handleRoot);
  server.on("/LED_on", handleLedOn);
  server.on("/LED_off", handleLedOff);
  server.on("/LED_toggle", handleLedToggle);
  server.on("/door_open", handleDoorOpen);
  server.on("/door_close", handleDoorClose);
  server.on("/temperature_req", handleTemperature);
  server.on("/humidity_req", handleHumidity);
  server.on("/gas_req", handleGas);
  server.on("/PIR_req", handlePIR);
  server.on("/Music_01", handleMusic);
  server.on("/music_stop", handleMusicStop);
  
  // Запуск сервера
  server.begin();
  Serial.println("Сервер запущен");
  Serial.println("\nПодключитесь к WiFi: " + String(ssid));
  Serial.println("Пароль: " + String(password));
  Serial.println("Откройте браузер: http://192.168.4.1");
  
  lcd.clear();
}

void loop() {
  // Обработка HTTP запросов
  server.handleClient();
  
  // Авто режим - возврат из WiFi
  if ((wifi_time != 0) && ((millis() - wifi_time) > 5000)) {
    lcd.setCursor(0, 1);
    lcd.print("Авто режим      ");
    wifi_mark = 0;
    wifi_time = 0;
  }
  
  // Обновление датчиков каждые 500мс
  if (millis() - lastUpdate > 500) {
    lastUpdate = millis();
    
    if (wifi_mark == 0) {
      checkButton();
      checkPIR();
      readClimate();
      checkGas();
      checkNFC();
    }
  }
}
